function [proc_data] = classified(data,punc_data,pre,post)
%This function takes data generated by punc_distill and performs a channel
%by channel classification based on distance.
%Synatax:   [classes] = classified(data,[1 2 4 7],[3 5 6]); 
%Input:     data = the data table generated by preclassify
%           punc_data = the original punc_data data structure
%           pre = all the presynaptic channels.  Enter 0 if there is no
%           pre.
%           post = all the postsynaptic channels.
%Output:    proc_data = the data generated through classification.  Fields:
%               classes = a 2D array, where each row is a base vector and 
%                       each column is a channel.  Values are 1 and 0, 
%                       where 1 is thereis a co-localized puncta, and 
%                       0 = not.
%               sum_data = a matrix, depth = 2(ori & termi), median and std
%                       of the distances of each channel from ori and termi
%               data = the final dataset used for the classification, with 
%                      not qualifying data removed.
%               ch_num = number of channels
%               ch_ord = the order of the channels
%               pre = presynaptic channels.
%               post = postsynpatic channels.
%               termi = the termini channel
%               ori = the origin channel

%parse the data structure first
if nargin == 3      %if only pre, no post
    post = 0;
end
[data,ori,termi,pre,post,ch_num,ch_ord,sum_data,col_loco] = parse(data,punc_data,pre,post);

%now begins the fun.
classes = false(size(data,1),ch_num);  %initiate the class array, it is logical
for i = 1:size(data,1)      %step through every base vector
    %grab a couple of needed data bits
    base_dist = data(i,11); %distance of termi from ori
    for j = 1:ch_num        %go through every channel
        if max(ch_ord(j)==pre)  %is this channel presynaptic or post synaptic?
            pos = 0;        %the channel is presynaptic
        elseif max(ch_ord(j)==post)
            pos = 1;        %this channel is post synaptic
        end
        switch pos
            case 0  %presynaptic
                if data(i,col_loco(1,j))<=base_dist  %within criteria
                    classes(i,j) = 1;       %true
                end
            case 1  %post synaptic
                if data(i,col_loco(2,j))<=(base_dist/2)   %within criteria
                    classes(i,j) = 1;       %true
                end
        end
    end
end

%create output structure
proc_data(1).classes = classes;
proc_data(1).data = data;
proc_data(1).sum_data = sum_data;
proc_data(1).ch_num = ch_num;
proc_data(1).ch_ord = ch_ord;
proc_data(1).pre = pre;
proc_data(1).post = post;
proc_data(1).ori = ori;
proc_data(1).termi = termi;

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [data,ori,termi,pre,post,ch_num,ch_ord,sum_data,col_loco] = parse(data,punc_data,pre,post)
%ch_num = the number of channels in the dataset minus ori & termi.
%sum_data = a matrix, depth = 2(ori & termi), median and std of the
%   distances of each channel from ori and termi.

%first create the final data structure
ch_num = (size(data,2)-12)/11;  %parse out the number of channels - ori and termi
[data,ch_ord] = preclassify(data,punc_data,ch_num);

%now pull out some relavent data
ori = punc_data(1).ori;
termi = punc_data(1).termi;

%remove ori and termi from pre and post if they are in there
pre = pre(pre~=ori); post = post(post~=ori);
pre = pre(pre~=termi); post = post(post~=termi);

%now filter the dataset so that we only index real base vectors
[y,x] = find(isnan(data(:,11)));    %remove oris without termi
data(y,:) = [];                     %gone
median_base = median(data(:,11));   %the median of the base vector length
std_base = std(data(:,11));         %the standard deviation of the base vector length
[y,x] = find(data(:,11)>repmat((median_base+std_base),size(data,1),1));    %remove vector lengths 1 standard deviation above the median
data(y,:) = [];                     %gone

%now generate the median and std array.
sum_data = single(zeros(2,ch_num,2));   %prepare the receiving array
for i = 1:ch_num
    curr_col = 22+((i-1)*13);       %first pre distance column
    pre_tmp = data(:,curr_col);     %grab all the data for the pre column
    post_tmp = data(:,curr_col+2);  %grab all the data for the post column
    pre_tmp(isnan(pre_tmp)) = [];   %delete empty data
    post_tmp(isnan(post_tmp)) = [];
    %store the calculated data
    sum_data(1,i,1) = median(pre_tmp);
    sum_data(2,i,1) = std(pre_tmp);
    sum_data(1,i,2) = median(post_tmp);
    sum_data(2,i,2) = std(post_tmp);
    col_loco(1,i) = curr_col;           %store the column location for later use.
end
%now calculate the col_loco for post channels
col_loco(2,:) = col_loco(1,:)+2;